"""
Author:         David Szybala
Website:        www.davidszybala.com
Alt Website:    www.twistedgerbil.com
Program:        unit_converter.py
Date:           11/13/14
Description:    This script will convert various units.
Installation:   Place this file in your maya scripts directory
                (e.g. ..\My Documents\maya\scripts) then run the following
                python commands:
                    import unit_converter
                    unit_converter.main()
"""
import maya.cmds as cmds


class UnitConverter(object):
    def __init__(self):
        self.unit_type = None
        self.from_unit_type = None
        self.to_unit_type = None
        self.from_unit_float = None
        self.to_unit_float = None
        self.from_unit_layout = None
        self.to_unit_layout = None
        self.unit_labels = {1: ['Millimeter', 'Centimeter', 'Meter', 'Kilometer', 'Inch', 'Feet', 'Yard', 'Mile'],
                            2: ['Celsius', 'Farrenheit', 'Kelvin'],
                            3: ['Gram', 'Kilogram', 'Ounce', 'Pound', 'Ton'],
                            4: ['Millisecond', 'Second', 'Minute', 'Hour', 'Day', 'Week', 'Month', 'Year'],
                            5: ['Milliliter', 'Liter', 'Teaspoon', 'Tablespoon', 'Ounce', 'Cup', 'Pint', 'Quart', 'Gallon']}
        self.length_units = {1: [1.0, 0.1, 0.001, 0.000001, 0.039370079, 0.00328084, 0.001093613, 0.000000621], 
                             2: [10.0, 1.0, 0.01, 0.00001, 0.393700787, 0.032808399, 0.010936133, 0.000006214], 
                             3: [1000.0, 100.0, 1.0, 0.001, 39.37007874, 3.280839895, 1.093613298, 0.00062137], 
                             4: [1000000.0, 100000.0, 1000.0, 1.0, 39370.078740158, 3280.839895013, 1093.613298338, 0.621369949], 
                             5: [25.4, 2.54, 0.0254, 0.0000254, 1, 0.083333333, 0.027777778, 0.000015783], 
                             6: [304.8, 30.48, 0.3048, 0.0003048, 12.0, 1.0, 0.333333333, 0.000189394], 
                             7: [914.4, 91.44, 0.9144, 0.0009144, 36.0, 3.0, 1.0, 0.000568181], 
                             8: [1609347.2186944, 160934.72186944, 1609.347218694, 1.609347219, 63360.126720253, 5280.010560021, 1760.003520007, 1.0]}
        self.mass_units = {1: [1.0, 0.001, 0.035273962, 0.002204623, 0.000001102], 
                           2: [1000.0, 1.0, 0.028349523, 0.45359237, 0.001102311], 
                           3: [28.349523125, 0.028349523, 1.0, 0.0625, 0.00003125], 
                           4: [453.59237, 0.45359237, 16.0, 1.0, 0.0005], 
                           5: [907184.74, 907.18474, 32000.0, 2000.0, 1.0]}
        self.time_units = {1: [1.0, 0.001, 0.000016667, 0.000000278, 0.000000000001, 0.000000000001, 0.000000000001, 0.000000000001], 
                           2: [1000.0, 1.0, 0.016666667, 0.000277778, 0.000011574, 0.000001653, 0.000000000001, 0.000000000001], 
                           3: [60000.0, 60.0, 1.0, 0.016666667, 0.000694444, 0.000099206, 0.000022831, 0.000001903], 
                           4: [3600000.0, 3600.0, 60.0, 1.0, 0.041666667, 0.005952381, 0.001369863, 0.000114155], 
                           5: [0.000000000001, 86400.0, 1440.0, 24.0, 1.0, 0.142857143, 0.032876712, 0.002739726], 
                           6: [0.000000000001, 604800.0, 10080.0, 168.0, 7.0, 1.0, 0.230136986, 0.019178082], 
                           7: [0.000000000001, 0.000000000001, 43800.0, 730.0, 30.416666667, 4.345238095, 1, 0.083333333], 
                           8: [0.000000000001, 0.000000000001, 525600.0, 8760.0, 365.0, 52.142857143, 12.0, 1.0]}
        self.volume_units = {1: [1.0, 0.001, 0.20288413621, 0.067628045405, 0.033814022701, 0.0042267528377, 0.0021133764189, 0.0010566882094, 0.00026417205236], 
                             2: [1000.0, 1.0, 202.88413621, 67.628045405, 33.814022701, 4.2267528377, 2.1133764189, 1.0566882094, 0.26417205236], 
                             3: [4.9289215938, 0.0049289215938, 1.0, 0.33333333334, 0.16666666667, 0.020833333334, 0.010416666667, 0.0052083333334, 0.0013020833333], 
                             4: [14.786764781, 0.014786764781, 2.9999999999, 1.0, 0.49999999998, 0.062499999999, 0.031249999999, 0.015625, 0.0039062499999], 
                             5: [29.573529563, 0.029573529563, 6.0, 2.0, 1.0, 0.125, 0.062500000001, 0.03125, 0.0078125000001], 
                             6: [236.5882365, 0.2365882365, 48.0, 16.0, 8.0, 1.0, 0.5, 0.25, 0.0625], 
                             7: [473.176473, 0.473176473, 96.0, 32.0, 16.0, 2.0, 1.0, 0.5, 0.125], 
                             8: [946.352946, 0.946352946, 192.0, 64.0, 32.0, 4.0, 2.0, 1.0, 0.25], 
                             9: [3785.411784, 3.785411784, 768.0, 256.0, 128.0, 16.0, 8.0, 4.0, 1.0]}        

    def get_length_conversion(self, from_unit, to_unit, from_value):
        """Returns a conversion value for length units"""
        result = from_value / self.length_units[to_unit][from_unit - 1]
        return result

    @staticmethod
    def get_temp_conversion(from_unit, to_unit, from_value):
        """Returns a conversion value for temperature units"""
        result = from_value

        if from_unit == to_unit:
            return result

        # Put everything in celsius
        if from_unit == 2:
            result = (from_value - 32) * 0.555556
        elif from_unit == 3:
            result = from_value - 273.15
        
        # Now convert
        if to_unit == 2:
            result = (result * 1.8) + 32
        elif to_unit == 3:
            result = result + 273.15
        
        return result

    def get_mass_conversion(self, from_unit, to_unit, from_value):
        """Returns a conversion value for mass units"""
        result = from_value / self.mass_units[to_unit][from_unit - 1]
        return result

    def get_time_conversion(self, from_unit, to_unit, from_value):
        """Returns a conversion value for time units"""
        result = from_value / self.time_units[to_unit][from_unit - 1]
        return result

    def get_volume_conversion(self, from_unit, to_unit, from_value):
        """Returns a conversion value for volume units"""
        result = from_value / self.volume_units[to_unit][from_unit - 1]
        return result

    def get_unit_conversion(self, unit_type, from_unit, to_unit, from_value):
        """Returns the current unit conversion value"""
        result = 0
        
        if unit_type == 1:
            result = self.get_length_conversion(from_unit, to_unit, from_value)
        elif unit_type == 2:
            result = self.get_temp_conversion(from_unit, to_unit, from_value)
        elif unit_type == 3:
            result = self.get_mass_conversion(from_unit, to_unit, from_value)
        elif unit_type == 4:
            result = self.get_time_conversion(from_unit, to_unit, from_value)
        else:
            result = self.get_volume_conversion(from_unit, to_unit, from_value)

        return result

    def set_unit_conversion(self, *args):
        """Unit and value changed event handler. Updates the unit with the
        new unit conversion"""
        current_unit = cmds.optionMenu(self.unit_type, query=True, select=True)
        from_value = cmds.floatField(self.from_unit_float, query=True, value=True)
        from_unit = cmds.optionMenu(self.from_unit_type, query=True, select=True)
        to_unit = cmds.optionMenu(self.to_unit_type, query=True, select=True)
        result = self.get_unit_conversion(current_unit, from_unit, to_unit, from_value)
        cmds.floatField(self.to_unit_float, edit=True, value=result)

    def set_unit_type(self, *args):
        """Unit changed event handler. Changes the available unit types"""
        current_unit = cmds.optionMenu(self.unit_type, query=True, select=True)
        
        cmds.deleteUI(self.from_unit_type)
        self.from_unit_type = cmds.optionMenu(width=100, parent=self.from_unit_layout,
                                              changeCommand=self.set_unit_conversion)
        for current in self.unit_labels[current_unit]:
            cmds.menuItem(label=current)
            
        cmds.deleteUI(self.to_unit_type)
        self.to_unit_type = cmds.optionMenu(width=100, parent=self.to_unit_layout,
                                            changeCommand=self.set_unit_conversion)
        for current in self.unit_labels[current_unit]:
            cmds.menuItem(label=current)
        
        cmds.optionMenu(self.to_unit_type, edit=True, select=2)
        
        self.set_unit_conversion()

    def setup_ui(self):
        """Creates the UI"""
        if cmds.window("unitConverterWindow", exists=True):
            cmds.deleteUI("unitConverterWindow")

        unit_window = cmds.window("unitConverterWindow", title="Unit Converter",
                                  minimizeButton=False, maximizeButton=False,
                                  sizeable=False, resizeToFitChildren=True)
        cmds.columnLayout()
        cmds.rowLayout(numberOfColumns=2, columnWidth2=[100, 100])
        cmds.text(width=100, label="Unit Type:", align="right")
        self.unit_type = cmds.optionMenu(width=100, changeCommand=self.set_unit_type)
        cmds.menuItem(label="Length")
        cmds.menuItem(label="Temp")
        cmds.menuItem(label="Mass")
        cmds.menuItem(label="Time")
        cmds.menuItem(label="Liquid")
        cmds.setParent("..")  # end rowLayout
        self.from_unit_layout = cmds.rowLayout(numberOfColumns=2, columnWidth2=[100,100])
        self.from_unit_float = cmds.floatField(width=100, minValue=(-999999999),
                                               maxValue=999999999, value=1.0,
                                               step=1.0, precision=6,
                                               changeCommand=self.set_unit_conversion)
        self.from_unit_type = cmds.optionMenu(width=100, changeCommand=self.set_unit_conversion)
        cmds.setParent("..")  # end rowLayout
        self.to_unit_layout = cmds.rowLayout(numberOfColumns=2, columnWidth2=[100,100])
        self.to_unit_float = cmds.floatField(width=100, minValue=(-999999999),
                                             maxValue=999999999, value=1.0,
                                             step=1.0, precision=6, editable=False)
        self.to_unit_type = cmds.optionMenu(width=100, changeCommand=self.set_unit_conversion)
        cmds.setParent("..")  # end rowLayout
        cmds.setParent("..")  # end columnLayout
        self.set_unit_type()

        cmds.showWindow(unit_window)


def main():
    """Main entry point. Opens the UI"""
    uc = UnitConverter()
    uc.setup_ui()


if __name__ == "__main__":
    main()
